import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { ShieldCheck, TrendingUp, Users } from "lucide-react"
import { Button } from "@/components/ui/button"
import { fetchApi } from "@/lib/api-client"
import Link from "next/link"

export async function ForumSidebar() {
  let trendingTags: string[] = []
  let guidelines: any = {
    title: "Community Guidelines",
    text: "This is a safe space for patients and caregivers.",
    bullets: [
      "Be empathetic and respectful.",
      "No medical advice - share experiences only.",
      "Clearly tag your cancer type and stage.",
      "Report misinformation immediately."
    ]
  }

  try {
    const [tagsData, contentData] = await Promise.all([
      fetchApi('tags.php'),
      fetchApi('content.php?page=forum')
    ])
    trendingTags = tagsData
    if (contentData.sidebar?.guidelines) {
      guidelines = JSON.parse(contentData.sidebar.guidelines)
    }
  } catch (error) {
    console.error("Failed to fetch sidebar data:", error)
    trendingTags = ["#BreastCancer", "#Immunotherapy", "#SideEffects", "#Chemotherapy", "#Nutrition"]
  }

  return (
    <div className="space-y-6">
      <Card className="border-primary/20">
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <ShieldCheck className="h-5 w-5 text-primary" />
            {guidelines.title}
          </CardTitle>
        </CardHeader>
        <CardContent className="text-sm space-y-4">
          <p>{guidelines.text}</p>
          <ul className="list-disc pl-4 space-y-2 text-muted-foreground">
            {guidelines.bullets?.map((bullet: string, i: number) => (
              <li key={i}>{bullet}</li>
            ))}
          </ul>
          <Button variant="outline" className="w-full text-xs h-8 bg-transparent" asChild>
            <Link href="/moderation">Read Full Guidelines</Link>
          </Button>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <TrendingUp className="h-5 w-5 text-secondary" />
            Trending Tags
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {trendingTags.map((tag) => (
              <Link key={tag} href={`/forum?q=${encodeURIComponent(tag.startsWith('#') ? tag.slice(1) : tag)}`}>
                <Badge
                  variant="secondary"
                  className="cursor-pointer hover:bg-primary hover:text-primary-foreground transition-colors"
                >
                  {tag}
                </Badge>
              </Link>
            ))}
          </div>
        </CardContent>
      </Card>

      <Card className="bg-primary/5 border-none">
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Users className="h-5 w-5 text-primary" />
            Join our Experts
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <p className="text-xs text-muted-foreground">
            Are you a healthcare professional? Apply for a Verified Expert badge to help guide our community with
            trusted insights.
          </p>
          <Button size="sm" className="w-full" asChild>
            <Link href="/profile">Apply for Verification</Link>
          </Button>
        </CardContent>
      </Card>
    </div>
  )
}
