"use client"

import Link from "next/link"
import { HeartPulse, PhoneCall } from "lucide-react"
import { useState, useEffect } from "react"
import { fetchApi } from "@/lib/api-client"

export function Footer() {
  const [crisisContent, setCrisisContent] = useState<any>({})
  const [footerContent, setFooterContent] = useState<any>({})

  useEffect(() => {
    Promise.all([
      fetchApi('content.php?page=crisis'),
      fetchApi('content.php?page=footer')
    ]).then(([crisis, footer]) => {
      setCrisisContent(crisis)
      setFooterContent(footer)
    }).catch(console.error)
  }, [])

  const nciPhone = crisisContent.nci?.phone || "1-800-4-CANCER (1-800-422-6237)"
  const nciPhoneDigits = nciPhone.match(/\d+/g)?.join('') || '18004226237'

  const resourceLinks = footerContent.resources?.links ? JSON.parse(footerContent.resources.links) : [
    { name: "Treatment Guides", href: "/guides" },
    { name: "Clinical Trials Navigator", href: "/trials" },
    { name: "Talking to Your Doctor", href: "/doctor-talk" },
    { name: "Financial Aid Resources", href: "/financial" }
  ]

  const aboutLinks = footerContent.about?.links ? JSON.parse(footerContent.about.links) : [
    { name: "Privacy Policy", href: "/privacy" },
    { name: "Terms of Service", href: "/terms" },
    { name: "Moderation Guidelines", href: "/moderation" },
    { name: "Contact Us", href: "/contact" }
  ]

  const disclaimerText = footerContent.disclaimer?.text || "Information on this site is for educational and support purposes only. It is not medical advice. Always consult with a qualified oncologist for diagnosis and treatment plans."

  return (
    <footer className="border-t bg-muted/30">
      <div className="container mx-auto px-4 py-12">
        <div className="grid grid-cols-1 md:grid-cols-4 gap-8">
          <div className="col-span-1 md:col-span-2 space-y-4">
            <div className="flex items-center gap-2">
              <HeartPulse className="h-6 w-6 text-primary" />
              <span className="text-lg font-bold text-primary">OncoConnect</span>
            </div>
            <p className="text-sm text-muted-foreground max-w-xs">
              Bridging the gap between medical expertise and community support. Empowering you with reliable information
              and a safe space for discussion.
            </p>
            <div className="bg-destructive/10 p-4 rounded-lg border border-destructive/20 max-w-sm">
              <div className="flex items-center gap-2 text-destructive font-semibold mb-1">
                <PhoneCall className="h-4 w-4" />
                <span>Need Immediate Support?</span>
              </div>
              <p className="text-xs text-destructive/80 mb-2">
                If you are in crisis or need urgent help, please contact the National Cancer Institute or a crisis
                hotline immediately.
              </p>
              <Link href={`tel:${nciPhoneDigits}`} className="text-xs font-bold underline text-destructive">
                Call NCI: {nciPhone}
              </Link>
            </div>
          </div>
          <div>
            <h3 className="font-semibold mb-4">Resources</h3>
            <ul className="space-y-2 text-sm text-muted-foreground">
              {resourceLinks.map((link: any) => (
                <li key={link.href}>
                  <Link href={link.href} className="hover:text-primary transition-colors">
                    {link.name}
                  </Link>
                </li>
              ))}
            </ul>
          </div>
          <div>
            <h3 className="font-semibold mb-4">About</h3>
            <ul className="space-y-2 text-sm text-muted-foreground">
              {aboutLinks.map((link: any) => (
                <li key={link.href}>
                  <Link href={link.href} className="hover:text-primary transition-colors">
                    {link.name}
                  </Link>
                </li>
              ))}
            </ul>
          </div>
        </div>
        <div className="mt-12 pt-8 border-t text-center">
          <p className="text-xs text-muted-foreground leading-relaxed">
            Disclaimer: {disclaimerText}
          </p>
          <p className="text-xs text-muted-foreground mt-4">
            © {new Date().getFullYear()} OncoConnect. All rights reserved.
          </p>
        </div>
      </div>
    </footer>
  )
}
