import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { FileText, Activity, Heart, ShieldCheck, LucideIcon, ArrowRight } from "lucide-react"
import { fetchApi } from "@/lib/api-client"
import Link from "next/link"

const ICON_MAP: Record<string, LucideIcon> = {
  FileText,
  Activity,
  Heart,
  ShieldCheck,
}

interface Guide {
  id: number;
  title: string;
  description: string;
  icon: string;
  color: string;
}

export default async function GuidesPage() {
  let guides: Guide[] = [];
  let content: any = {};
  try {
    guides = await fetchApi('guides.php');
    content = await fetchApi('content.php?page=guides');
  } catch (error) {
    console.error("Failed to fetch guides:", error);
  }

  return (
    <main className="flex-1">
      <section className="bg-primary/5 py-16">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl">
            <h1 className="text-4xl font-bold mb-4">{content.header?.title || "Patient & Caregiver Guides"}</h1>
            <p className="text-xl text-muted-foreground">
              {content.header?.description || "Empowering you with the knowledge to navigate your cancer journey with confidence."}
            </p>
          </div>
        </div>
      </section>

      <section className="container mx-auto px-4 py-12">
        <div className="grid md:grid-cols-2 gap-6">
          {guides.length > 0 ? guides.map((guide) => {
            const Icon = ICON_MAP[guide.icon] || FileText;
            return (
              <Card key={guide.id} className="hover:shadow-md transition-shadow group overflow-hidden">
                <Link href={`/guides/${guide.id}`}>
                  <CardHeader className="flex flex-row items-center gap-4">
                    <div className={`p-2 rounded-lg bg-muted ${guide.color}`}>
                      <Icon className="h-6 w-6" />
                    </div>
                    <CardTitle className="group-hover:text-primary transition-colors">{guide.title}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-muted-foreground mb-6">{guide.description}</p>
                    <div className="flex items-center text-primary font-bold text-sm gap-1">
                      Read Full Guide <ArrowRight className="h-4 w-4 group-hover:translate-x-1 transition-transform" />
                    </div>
                  </CardContent>
                </Link>
              </Card>
            );
          }) : (
            <div className="col-span-2 p-12 text-center bg-background rounded-xl border">
              <p className="text-muted-foreground italic">No guides found.</p>
            </div>
          )}
        </div>
      </section>
    </main>
  )
}
