import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { ExternalLink, ShieldCheck, Heart } from "lucide-react"
import { fetchApi } from "@/lib/api-client"

interface FinancialResource {
  id: number;
  title: string;
  description: string;
  link: string;
}

export default async function FinancialAidPage() {
  let resources: FinancialResource[] = [];
  try {
    resources = await fetchApi('financial.php');
  } catch (error) {
    console.error("Failed to fetch financial resources:", error);
  }

  return (
    <main className="flex-1">
      <section className="bg-primary/5 py-16">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl">
            <h1 className="text-4xl font-bold mb-4">Financial Aid Resources</h1>
            <p className="text-xl text-muted-foreground">
              Navigating the costs of cancer treatment can be overwhelming. We've gathered trusted resources to help you
              find support.
            </p>
          </div>
        </div>
      </section>

      <section className="container mx-auto px-4 py-12">
        <div className="grid md:grid-cols-2 gap-8 items-start">
          <div className="space-y-6">
            <h2 className="text-2xl font-bold">Recommended Organizations</h2>
            <div className="grid gap-4">
              {resources.length > 0 ? resources.map((resource) => (
                <Card key={resource.id} className="hover:shadow-md transition-shadow">
                  <CardHeader className="pb-2">
                    <CardTitle className="text-lg flex items-center justify-between">
                      {resource.title}
                      <ExternalLink className="h-4 w-4 text-muted-foreground" />
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-muted-foreground mb-4">{resource.description}</p>
                    <Button variant="outline" size="sm" asChild className="bg-transparent">
                      <a href={resource.link} target="_blank" rel="noopener noreferrer">
                        Visit Website
                      </a>
                    </Button>
                  </CardContent>
                </Card>
              )) : (
                <p className="text-muted-foreground italic">No resources found.</p>
              )}
            </div>
          </div>

          <div className="space-y-6">
            <Card className="bg-secondary/10 border-none">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <ShieldCheck className="h-5 w-5 text-secondary" />
                  Tips for Financial Planning
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4 text-sm">
                <p>1. Contact your hospital's billing department early to discuss payment plans.</p>
                <p>2. Ask to speak with an oncology social worker about local grants and assistance.</p>
                <p>3. Check if your pharmaceutical company has a Patient Assistance Program (PAP).</p>
                <p>4. Keep detailed records of all medical expenses for potential tax deductions.</p>
              </CardContent>
            </Card>

            <div className="p-6 rounded-xl border border-primary/20 bg-primary/5 flex items-start gap-4">
              <div className="p-2 rounded-lg bg-primary/10">
                <Heart className="h-6 w-6 text-primary" />
              </div>
              <div>
                <h3 className="font-bold mb-1">Community Support</h3>
                <p className="text-sm text-muted-foreground mb-3">
                  Our forum has a dedicated #FinancialAid tag where members share their experiences with different
                  programs.
                </p>
                <Button variant="link" className="p-0 h-auto font-bold text-primary" asChild>
                  <a href="/forum?search=FinancialAid">Browse Discussions →</a>
                </Button>
              </div>
            </div>
          </div>
        </div>
      </section>
    </main>
  )
}
