import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { ClipboardList, HelpCircle } from "lucide-react"
import { fetchApi } from "@/lib/api-client"

interface ChecklistItem {
  id: number;
  category: 'before' | 'during' | 'after';
  task: string;
}

interface QuestionItem {
  id: number;
  question: string;
}

export default async function DoctorTalkPage() {
  let checklists: ChecklistItem[] = [];
  let questions: QuestionItem[] = [];

  try {
    const data = await fetchApi('doctor-talk.php');
    checklists = data.checklists || [];
    questions = data.questions || [];
  } catch (error) {
    console.error("Failed to fetch doctor-talk data:", error);
  }

  const beforeTasks = checklists.filter(item => item.category === 'before');
  const duringTasks = checklists.filter(item => item.category === 'during');

  return (
    <main className="flex-1">
      <section className="bg-secondary/10 py-16">
        <div className="container mx-auto px-4">
          <h1 className="text-4xl font-bold mb-4">Talking to Your Doctor</h1>
          <p className="text-xl text-muted-foreground max-w-2xl">
            Effective communication with your oncology team is crucial. Use these resources to prepare for your next
            appointment.
          </p>
        </div>
      </section>

      <section className="container mx-auto px-4 py-12 space-y-12">
        <div className="grid md:grid-cols-3 gap-8">
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ClipboardList className="h-5 w-5 text-primary" />
                Appointment Checklist
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <p className="font-semibold italic text-sm">Before the appointment:</p>
                <ul className="list-disc pl-5 space-y-1 text-muted-foreground text-sm">
                  {beforeTasks.map(item => (
                    <li key={item.id}>{item.task}</li>
                  ))}
                  {beforeTasks.length === 0 && <li>No tasks found.</li>}
                </ul>
              </div>
              <div className="space-y-2">
                <p className="font-semibold italic text-sm">During the appointment:</p>
                <ul className="list-disc pl-5 space-y-1 text-muted-foreground text-sm">
                  {duringTasks.map(item => (
                    <li key={item.id}>{item.task}</li>
                  ))}
                  {duringTasks.length === 0 && <li>No tasks found.</li>}
                </ul>
              </div>
            </CardContent>
          </Card>

          <Card className="bg-primary text-primary-foreground">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <HelpCircle className="h-5 w-5" />
                Key Questions to Ask
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4 text-sm opacity-90">
              {questions.map((item, index) => (
                <p key={item.id}>{index + 1}. {item.question}</p>
              ))}
              {questions.length === 0 && <p>No questions found.</p>}
            </CardContent>
          </Card>
        </div>
      </section>
    </main>
  )
}
